﻿using System.Collections.Generic;
using System.Linq;

using HIPS.Web.Components.Common;

namespace HIPS.Web.Components.ServiceModel
{
    /// <summary>
    /// Represents a list of textual response messages.
    /// </summary>
    /// <history>
    ///   <change user="David Sampson (Chamonix)" date="23 November 2013">Initial version.</change>
    /// </history>
    public class ResponseMessageList : List<ResponseMessage>
    {

        #region Methods

        /// <summary>
        /// Adds a new message to the list.
        /// </summary>
        /// <param name="description">Description text for the message.</param>
        /// <param name="level">Level associated with the message.</param>
        /// <history>
        ///   <change user="David Sampson (Chamonix)" date="18 December 2013">Initial version.</change>
        /// </history>
        public void Add(string description, MessageLevel level)
        {
            this.Add(new ResponseMessage(description, level));
        }

        /// <summary>
        /// Returns the messages in this list that have an Error MessageLevel.
        /// </summary>
        public List<ResponseMessage> Errors
        {
            get { return this.Where(m => m.Level == MessageLevel.Error).ToList(); }
        }

        /// <summary>
        /// Returns the messages in this list that have a Warning MessageLevel.
        /// </summary>
        public List<ResponseMessage> Warnings
        {
            get { return this.Where(m => m.Level == MessageLevel.Warning).ToList(); }
        }

        /// <summary>
        /// Returns the messages in this list that have an Information MessageLevel.
        /// </summary>
        public List<ResponseMessage> Informations
        {
            get { return this.Where(m => m.Level == MessageLevel.Information).ToList(); }
        }

        /// <summary>
        /// Determines if the list of messages contains any warning messages.
        /// </summary>
        /// <returns>Boolean. true if the list contains any warning messages, false otherwise.</returns>
        /// <history>
        ///   <change user="David Sampson (Chamonix)" date="23 November 2013">Initial version.</change>
        /// </history>
        public bool ContainsWarning()
        {
            return this.Any(m => m.Level == MessageLevel.Warning);
        }

        /// <summary>
        /// Determines if the list of messages contains any error messages.
        /// </summary>
        /// <returns>Boolean. true if the list contains any error messages, false otherwise.</returns>
        /// <history>
        ///   <change user="David Sampson (Chamonix)" date="23 November 2013">Initial version.</change>
        /// </history>
        public bool ContainsError()
        {
            return this.Any(m => m.Level == MessageLevel.Error);
        }

        /// <summary>
        /// Return a string representation of the messages in the list.
        /// </summary>
        /// <returns>String.</returns>
        /// <history>
        ///   <change user="David Sampson (Chamonix)" date="03 February 2014">Initial version.</change>
        /// </history>
        public string AsString()
        {
            return string.Join(" ", this.Select(m => m.Description));
        }

        #endregion

    }
}